clear
close all
clc
addpath("functions\");

%% Load colours
blue = importdata("blue_palette.txt");
for i = 1 : height(blue)
    blue(i, :) = blue(i, :) / 255;
end

red = importdata("red_palette.txt");
for i = 1 : height(red)
    red(i, :) = red(i, :) / 255;
end

%% INPUT DATA

opts.catchment = "biois";
opts.label = "BS";
opts.shapefile = "%CATCHMENT%\networkSplit.shp";

addpath(strrep("D:\2 - Data availability\1 - Catchments\%CATCHMENT%", "%CATCHMENT%", opts.catchment));

stretches = struct2table(shaperead(strrep("%CATCHMENT%\networkSplit.shp", "%CATCHMENT%", opts.catchment)));
Pi = stretches.Pi;  % local persistency
AD8 = stretches.ad8 * 1e-6; % [km2] contributing area
D8 = readgeoraster(strrep("%CATCHMENT%\p.tif", "%CATCHMENT%", opts.catchment)); %% turbolo: DTMp_cor.tif, altri p.tif
[dtm, ref] = readgeoraster(strrep("%CATCHMENT%\dtm.tif", "%CATCHMENT%", opts.catchment));  %% turbolo: dtmfel

side = ref.CellExtentInWorldX * 1e-3;   % [km]
Lnet = sum(stretches.length) * 1e-3;   % [km] extracted from the DTM
Li = stretches.length * 1e-3;           % [km]


%% Define threshold areas
int = 300;
A_inf = max(AD8) * 0.001; % 0.1% of the maximum area of the catchment
A_sup = max(AD8) * 0.999; % 99.9% of the maximum area of the catchment

Astar = linspace((A_inf), (A_sup), int)';

%% Calculcation of temporary fraction = PHI

Ti = 1 - Pi;        % local temporariness

BPi = zeros(size(AD8));  % Binary Persisentcy

for i = 1: length(AD8)
    if Pi(i) >= 0.99
        BPi(i) = 1;
    else 
        BPi(i) = 0;
    end
end

BTi = 1 - BPi;      % Binary Temporariness

% Calculation of phi for Astar
Tf = NaN(height(Astar), 1);

for j = 1 : height(Astar)
    
    % upstream
    k = AD8 < Astar(j);
    Tf(j) = (sum(BTi(k) .* Li(k))) / (sum(Li(k)));

end

PHI = Tf;                  % phi definition
PHI(isnan(PHI)) = 1;       % remove NaN values


%% Calculation of the total, perennial and temporary length

% Total Length fo the network extracted form the DTM
L = NaN(int, 1);
Lw = NaN(int, 1);
Ld = NaN(int, 1);

for i = 1 : int

    % Total Length
    rete = AD8 > Astar(i);
    D8_rete = D8(rete);
    diagonal = rem(D8_rete,2) == 0;
    L_result = sum(diagonal) * sqrt(2) * side + sum(~diagonal) * side;
    L(i) = L_result;

    % Wet Length
    righeW = find(Ti <= 0.01);    % perennial nodes
    AD8wet = AD8(righeW);
    retewet = AD8wet > Astar(i);
    D8_retewet = D8(retewet);
    diagonalwet = rem(D8_retewet, 2) == 0;
    Lw_result = sum(diagonalwet) * sqrt(2) * side + sum(~diagonalwet) * side;
    Lw(i) = Lw_result;

    % Dry Length
    righeD = find(Ti > 0.01);    % temporary nodes
    AD8dry = AD8(righeD);
    retedry = AD8dry > Astar(i);
    D8_retedry = D8(retedry);
    diagonaldry = rem(D8_retedry, 2) == 0;
    Ld_result = sum(diagonaldry) * sqrt(2) * side + sum(~diagonaldry) * side;
    Ld(i) = Ld_result;

end


%% CALIBRATION OF A and B over the TOTAL LENGTH

% Removal of the finite size effect tail
percent_area = 0.8;
id = Astar >= percent_area * max(Astar);
h_test = length(L(~id));
L_test = L(~id);
Astar_test = Astar(~id);

h = length(L_test);
Rsquare_result = NaN(h, h);
A = NaN(h,h);   % matrix a of the power law
B = NaN(h,h);   % matrix b of the power law
N = NaN(h,h);   % difference between k and i

percent_points = 0.3;
n_points = h*percent_points;   % min number of points included

for i = 1 : h_test - n_points
    for k = (i + n_points - 1) : h_test

        N(i,k) = k-i;

        X = log(Astar_test((i:k),1));
        Y = log(L_test((i:k),1));

        b = cov(X, Y);
        b = b(1, 2) / b(1, 1);  % Sxy / Sxx
        a = exp(mean(Y) - b*mean(X));
        A(i,k) = a;
        B(i,k) = b;

        Yreg = log(a) + b*X;
        RSS = sum((Y-Yreg).^2);
        TSS = sum((Y-mean(Y)).^2);
        Rsquare = 1 - RSS/TSS;
        
        Rsquare_result(i,k) = Rsquare;
    end
end

% Calculation of Amin for each possible combiantions of a and b
Amin_good = (Lnet ./ A) .^(1./B);   % B is negative

% Imposition of a threshold: 1) R2 > 0.95, 2) Amin > A_inf
threshold = Rsquare_result > 0.95;

A_good = A(threshold);
B_good = B(threshold);

threshold = (A_inf <= Amin_good(threshold));

A_good = A_good(threshold);
B_good = B_good(threshold);

B_good = abs(B_good); % B is positive

% Optimization of a and b
pA = ksdensity(A_good, A_good);
pB = ksdensity(B_good, B_good); 

likelihood = (pA .* pB)';
[~, i] = max(likelihood);

A_1 = A_good(i);
B_1 = B_good(i);
AMIN_1 = (Lnet ./ A_1) .^ (- 1/B_1);

% Find all the curves that interpolate the length data

Abig = 10.^(linspace(log10(10^-4), log10(10^7), int))';
x = Abig;
y = NaN(length(x), length(B_good));

for i = 1 : length(x)
    for j = 1 : length(B_good)
        y(i, j) = A_good(j) .* x(i) .^ (-B_good(j));    % B is positive number
    end
end

% Find the best curve that interpolates the points
y_best = A_1 .* x .^ (-B_1);
y_baby = A_1 .* Astar .^ (-B_1);

interval = abs(L-y_baby) < 0.1 * y_baby;    % points within 10% of the interpolation line

RSS = sum((L(interval) - y_baby(interval)).^2);
TSS = sum((L(interval) - mean(L(interval))).^2);
RsquareL = 1 - RSS/TSS;
               
%% Plot of interpolation line for the length data

figure()
    
    legendEntry = plot(NaN, NaN, '.','Color', 'none', 'DisplayName', opts.label);
    hold on
    % yline(Lnet, '-', 'LineWidth',2,'Color',color.black.rgb, 'DisplayName','L_{net}')
    hold on
    plot(x, y(:, 1:50:end), '-', 'LineWidth', 2,'Color', color.light_grey.rgb, 'HandleVisibility','off');
    legendEntry = plot(NaN, NaN, '-', 'LineWidth', 2,'Color', color.light_grey.rgb, ...
        'DisplayName', "Interpolation lines");
    hold on
    plot(Astar(1:1:end), L(1:1:end), 'o', 'MarkerSize', 4, 'MarkerFaceColor', blue(11,:), ...
        'MarkerEdgeColor', blue(12, :), 'LineWidth', 0.1, 'DisplayName', "Data");
    hold on
    % plot(Astar(interval), L(interval), '.', 'MarkerSize', 30, 'Color', color.red.rgb, 'DisplayName', "Best fitting data")
    hold on
    % plot(x, y_best, '-', 'LineWidth', 5,'Color', color.red.rgb, 'DisplayName', "Best fitting line: two steps");

    set(gca, "XScale", 'log');
    set(gca, "Yscale","log");
    set(gcf, "Units", "centimeters");
    set(gcf, 'Position', [3 3 9 6]);
    set(gca,'ClippingStyle','rectangle');
    box on

    xlabel('Contributing area (km^2)', 'FontSize', 8);
    ylabel('Total Length (km)', 'FontSize', 8);
    leg = legend('Location', 'northeast', 'Orientation','vertical');
    leg.FontSize = 8;
    leg.ItemTokenSize = [10, 10]; % [lunghezza, altezza]
    leg.Box = 'off';
    xlim([10^-2, 10^2]);
    ylim([10^-3, 10^4]);
   
    ax = gca;
    ax.FontSize = 8;
    ax.TickLength = [0.01, 0.01]; 
    ax.YTick = [10^-3, 10^4];

%% PARAMETER DEFINITIONS

Cmin = 0.01;
Cmax = 10*max(A_good);
Dmin =  0.01;
Dmax = 10*max(B_good);

N = 10^6;     % # values to test with Monte Carlo method

a = min(A_good) + (max(A_good) - min(A_good)) * rand(N,1);
b = min(B_good) + (max(B_good) - min(B_good)) * rand(N, 1);
c = Cmin + (Cmax-Cmin) * rand(N, 1);
d = Dmin + (Dmax-Dmin) * rand(N, 1);
    
%% TWO STEP METHOD where C and D are calibrated for each A, B and Amin in the interval

PARAM_TWOSTEPS = NaN(N, 9);

for i = 1 : N   % righe della matrice param_twosteps
    
    j = randi(length(A_good), 1);
    A = A_good(j);
    B = B_good(j);
    
    C = c(i);
    D = d(i);

    ATHRESHOLD = (C / A) .^(1/(D-B));
    AMIN = (Lnet/A).^(-1/B);
    K = C / A;
    GAMMA = B - D;

    PHIresult = NaN(numel(Astar), 1);
    
    % before Athreshold
    id = Astar < ATHRESHOLD;
    PHIresult(id) = (1 - K .* AMIN.^(GAMMA) .* ...
            (1 - (Astar(id)./AMIN).^(-D)) ./ (1 - (Astar(id)./AMIN).^(-B)));
    % after Athreshold
    id = ~id;
    PHIresult(id) = (1 - K .* AMIN .^(GAMMA)) ./ (1-(Astar(id)./AMIN).^(-B));
    
    % Calculation of R^2
    SS_res = sum((PHI - PHIresult).^2);
    SS_tot = sum((PHI - mean(PHI)).^2);
    Rsquare_result = 1 - (SS_res / SS_tot);

     % Saving the variables 
    PARAM_TWOSTEPS(i, :) = [A, B, C, D, AMIN, ATHRESHOLD, K, GAMMA, Rsquare_result];
    
end

PARAM_TWOSTEPS(any(isnan(PARAM_TWOSTEPS), 2), :) = []; % delete any NaN entries

A_3 = PARAM_TWOSTEPS(:,1);
B_3 = PARAM_TWOSTEPS(:, 2);
C_3 = PARAM_TWOSTEPS(:, 3);
D_3 = PARAM_TWOSTEPS(:, 4);
AMIN_3 = PARAM_TWOSTEPS(:, 5);
ATHRESHOLD_3 = PARAM_TWOSTEPS(:, 6);
K_3 = PARAM_TWOSTEPS(:, 7);
GAMMA_3 = PARAM_TWOSTEPS(:, 8);
R2_3 = PARAM_TWOSTEPS(:, 9);

% Selection of the best values
threshold = (R2_3 >= 0.95 * max(R2_3)) & (AMIN_3 <= ATHRESHOLD_3);
A_3_int = A_3(threshold);
B_3_int = B_3(threshold);
C_3_int = C_3(threshold);
D_3_int = D_3(threshold);
K_3_int = K_3(threshold);
GAMMA_3_int = GAMMA_3(threshold);
AMIN_3_int = AMIN_3(threshold);
ATHRESHOLD_3_int = ATHRESHOLD_3(threshold);
R2_3_int = R2_3(threshold);

% Finding the best combination
id = find(R2_3_int == max(R2_3_int));
A_3 = A_3_int(id);
B_3 = B_3_int(id);
C_3 = C_3_int(id);
D_3 = D_3_int(id);
K_3 = K_3_int(id);
GAMMA_3 = GAMMA_3_int(id);
AMIN_3 = AMIN_3_int(id);
ATHRESHOLD_3 = ATHRESHOLD_3_int(id);

Rsquare = max(R2_3_int);

%------------------------------------------------------------------------%
% LOCAL SCALE (on Astar) 
% Best case 3
id = Astar < ATHRESHOLD_3;
curve_3_baby(id) = (1 - K_3 .* AMIN_3.^(GAMMA_3) .* ...
        (1 - (Astar(id)./AMIN_3).^(-D_3)) ./ (1 - (Astar(id)./AMIN_3).^(-B_3)));
id = ~id;
curve_3_baby(id) = (1 - K_3 .* AMIN_3 .^(GAMMA_3)) ./ (1-(Astar(id)./AMIN_3).^(-B_3));
curve_3_baby = curve_3_baby';

% Interval 1
curve_3_int_baby = NaN(length(Astar), height(ATHRESHOLD_3_int));
for i = 1 : height(ATHRESHOLD_3_int)
    id = Astar < ATHRESHOLD_3_int(i);
    curve_3_int_baby(id, i) = 1 - K_3_int(i) * AMIN_3_int(i) ^ GAMMA_3_int(i) * (1 - (Astar(id)./AMIN_3_int(i)).^(-D_3_int(i))) ./ (1 - (Astar(id)./AMIN_3_int(i)).^(-B_3_int(i)));
    id = ~id;
    curve_3_int_baby(id, i) = (1 - (C_3_int(i)/A_3_int(i)) .* AMIN_3_int(i) .^ GAMMA_3_int(i)) ./ (1-(Astar(id)./AMIN_3_int(i)).^(-B_3_int(i)));
end

%------------------------------------------------------------------------%
% REGIONAL SCALE (on Abig)
% Best case 1
id = Abig < ATHRESHOLD_3;
curve_3(id) = (1 - K_3 .* AMIN_3.^(GAMMA_3) .* ...
        (1 - (Abig(id)./AMIN_3).^(-D_3)) ./ (1 - (Abig(id)./AMIN_3).^(-B_3)));
curve_datry(id) = 1 - K_3 .* Abig(id) .^(GAMMA_3);
id = ~id;
curve_3(id) = (1 - K_3 .* AMIN_3 .^(GAMMA_3)) ./ (1-(Abig(id)./AMIN_3).^(-B_3));
curve_datry(id) = zeros(length(Abig(id)), 1);
curve_3 = curve_3';

% Interval 1
curve_3_int_big = NaN(length(Astar), height(ATHRESHOLD_3_int));
for i = 1 : height(ATHRESHOLD_3_int)
    id = Abig < ATHRESHOLD_3_int(i);
    curve_3_int_big(id, i) = 1 - K_3_int(i) * AMIN_3_int(i) ^ GAMMA_3_int(i) * (1 - (Abig(id)./AMIN_3_int(i)).^(-D_3_int(i))) ./ (1 - (Abig(id)./AMIN_3_int(i)).^(-B_3_int(i)));
    id = ~id;
    curve_3_int_big(id, i) = (1 - K_3_int(i) .* AMIN_3_int(i) .^ GAMMA_3_int(i)) ./ (1-(Abig(id)./AMIN_3_int(i)).^(-B_3_int(i)));
end

%------------------------------------------------------------------------%
% Addition of the interpolation line with the best parameters
y_3 = A_3 .* Abig .^ (-B_3);
y_3_baby = A_3 .* Astar .^ (-B_3);

y_3_int = NaN(int, length(A_3_int));
for i = 1 : length(A_3_int)
    y_3_int(:, i) = A_3_int(i) .* Abig .^ (-B_3_int(i));
end

interval = abs(L-y_3_baby) < 0.1 * y_3_baby; 

RSS = sum((L(interval) - y_3_baby(interval)).^2);
TSS = sum((L(interval) - mean(L(interval))).^2);
RsquareL_3 = 1 - RSS/TSS;

%% LIKELIHOOD CALCULATION

% Likelihood of a and b
pA = ksdensity(A_good, A_good);
pB = ksdensity(B_good, B_good); 

likelihood_ab = (pA .* pB)';

% Likelihood of c and d
pC = ksdensity(C_3_int, C_3_int);
pD = ksdensity(D_3_int, D_3_int); 

likelihood_cd = (pC .* pD)';

% Plot of HISTOGRAMS + LIKELIHOOD
figure()

    set(gcf, "Units", "centimeters");
    set(gcf, 'Position', [3 3 18 5]);

    subplot(1,4,1)
    histogram(A_good, 10, "Normalization", "pdf", 'DisplayName', 'Histogram of a');
    hold on;
    [ya, xia] = ksdensity(A_good);
    plot(xia, ya, "LineWidth", 2, 'Color', color.red.rgb,"DisplayName","Pdf(a)");
    xlabel("Coefficient a", 'FontSize',8);
    ylabel("Frequency", 'FontSize',8);
    leg = legend('Location', 'southoutside', 'Orientation','horizontal');
    leg.FontSize = 8;
    leg.ItemTokenSize = [8, 8]; % [lunghezza, altezza]
    leg.Box = 'off';
    ax = gca;
    ax.FontSize = 8;
    ax.TickLength = [0.01, 0.01];
    xlim([0.7*min(A_good), 1.1*max(A_good)]);

    subplot(1,4,2)

    histogram(B_good, 10, "Normalization","pdf", 'DisplayName', 'Histogram of b');
    hold on;
    [yb, xib] = ksdensity(B_good);
    plot(xib, yb, "LineWidth", 2, 'Color', color.red.rgb,"DisplayName", "Pdf(b)");
    xlabel("Coefficient b", 'FontSize',8);
    ylabel("Frequency", 'FontSize',8);
    leg = legend('Location', 'southoutside', 'Orientation','horizontal');
    leg.FontSize = 8;
    leg.ItemTokenSize = [8, 8]; % [lunghezza, altezza]
    leg.Box = 'off';
    ax = gca;
    ax.FontSize = 8;
    ax.TickLength = [0.01, 0.01]; 
    xlim([0.95*min(B_good), 1.1*max(B_good)]);

    subplot(1,4,3)
    histogram(C_3_int, 10, "Normalization", "pdf", 'DisplayName', 'Histogram of c');
    hold on;
    [yc, xic] = ksdensity(C_3_int);
    plot(xic, yc, "LineWidth", 2, 'Color', color.red.rgb,"DisplayName","Pdf(c)");
    xlabel("Coefficient c", 'FontSize',8);
    ylabel("Frequency", 'FontSize',8);
    leg = legend('Location', 'southoutside', 'Orientation','horizontal');
    leg.FontSize = 8;
    leg.ItemTokenSize = [8, 8]; % [lunghezza, altezza]
    leg.Box = 'off';
    ax = gca;
    ax.FontSize = 8;
    ax.TickLength = [0.01, 0.01]; 
    xlim([0.7*min(C_3_int), 1.1*max(C_3_int)]);

    subplot(1,4,4)
    histogram(D_3_int, 10, "Normalization", "pdf", 'DisplayName', 'Histogram of d');
    hold on;
    [yd, xid] = ksdensity(D_3_int);
    plot(xid, yd, "LineWidth", 2, 'Color', color.red.rgb,"DisplayName","Pdf(d)");
    xlabel("Coefficient d", 'FontSize',8);
    ylabel("Frequency", 'FontSize',8);
    leg = legend('Location', 'southoutside', 'Orientation','horizontal');
    leg.FontSize = 8;
    leg.ItemTokenSize = [8, 8]; % [lunghezza, altezza]
    leg.Box = 'off';
    ax = gca;
    ax.FontSize = 8;
    ax.TickLength = [0.01, 0.01]; 
    xlim([0.8*min(D_3_int), 1.1*max(D_3_int)]);

%% INFORMATIVE CONTENT: 
% calculation of how informative PhiC and PhiN are at different spatial scales

id = find(Abig >= 0.95*AMIN_3, 1);  % intervallo (Amin, Amax)
Area_int = Abig(id:end);
curve_minor = curve_3(id:end);      % PhiC
curve_major = curve_datry(id:end);  % PhiN

IC = NaN(length(curve_major), 1);
for i = 1 : length(curve_minor)
    IC(i) = (curve_3(end) - curve_major(i)) / (curve_minor(i) - curve_major(i));
end

id = find(IC >= 0.9, 1);
ic90 = IC(id);
area90 = Area_int(id);

id = find(Area_int >= max(AD8), 1);
icCAT = IC(id);
areaCAT = Area_int(id);

IC_COMPL = 1 - IC;

% Plot of IC
figure()

    legendEntry = plot(NaN, NaN, '.','Color', 'none', 'DisplayName', opts.label);
    hold on
    plot(Area_int, IC, '-', 'LineWidth', 3, 'Color', color.black.rgb, 'HandleVisibility', 'off');
    hold on
    plot(Area_int, IC_COMPL, '-', 'LineWidth', 3, 'Color', color.grey.rgb, 'HandleVisibility', 'off');
    hold on
    legendEntry1 = plot(NaN, NaN, '-', 'LineWidth', 3, 'Color', color.black.rgb, 'DisplayName', 'ic' );
    hold on
    legendEntry2 = plot(NaN, NaN, '-', 'LineWidth', 3, 'Color', color.grey.rgb, 'DisplayName', '1 - ic' );
    hold on
    plot(areaCAT, icCAT, '*', 'DisplayName', 'ic(A_{max}) = IC');

    set(gca, 'XScale', 'log');
    xlim([AMIN_3, 10^2])
    ylim([0, 1])
    set(gcf, 'Position', [50 50 600 300]);

    xlabel('log (Contributing area) [km^2]', 'FontSize', 25);
    ylabel('\alpha', 'FontSize', 25);
    leg = legend('Location', 'east', 'Orientation','vertical');
    leg.FontSize = 15;
    leg.ItemTokenSize = [20, 20]; % [lunghezza, altezza]
    leg.Box = 'off';
    ax = gca;
    ax.FontSize = 15;
    ax.TickLength = [0.01, 0.01];


%% Plot of the TEMPORARY FRACTION

% Reducing the number of points displayed in the graphic
k = 60;
idx = logspace(log10(1), log10(length(PHI)), k);
Tf = PHI(round(idx));
Area = Astar(round(idx));


    figure()
    plot(Abig, curve_3_int_big, '-','Color', [0.9 0.9 0.9 0.7], 'LineWidth', 7, 'HandleVisibility','off');
    hold on
    legendEntry = plot(NaN, NaN, '.','Color', 'none', 'DisplayName', opts.label);
    hold on
    legendEntry1 = plot(NaN, NaN, '-','Color', [0.9 0.9 0.9], 'LineWidth', 5, 'DisplayName', 'CI');
    hold on
    plot(Abig, curve_3, '-','Color', color.dark_blue.rgb, 'LineWidth', 2, 'DisplayName',' \phi_< (Eq. (1))');
    hold on
    plot(Area, Tf, 'o', 'MarkerSize', 4, 'MarkerEdgeColor', color.dark_grey.rgb, ...
        'MarkerFaceColor', blue(5, :), 'LineWidth', 0.1, ...
         'DisplayName', 'Data');
    
    set(gca, "XScale", 'log');
    set(gca, "Layer", 'top');
    set(gca,'ClippingStyle','rectangle');
    
    set(gcf, "Units", "centimeters");
    set(gcf, 'Position', [3 3 10 3]); % figure for main
    % set(gcf, 'Position', [3 3 9 4]); % figure for SI

    xlabel('Contributing area (km^2)', 'FontSize', 5);
    ylabel('Temporary fraction', 'FontSize', 5);
    leg = legend('Location', 'southwest', 'Orientation','vertical');
    leg.FontSize = 5;
    leg.ItemTokenSize = [8, 8]; % [lunghezza, altezza]
    leg.Box = 'off';
    
    ax = gca;
    ax.FontSize = 5;
    ax.TickLength = [0.01, 0.01]; 
    
    xlim([10^-4, 10^2]);
    ylim([0, 1]);


%% ERRORS

% Standard Deviation of Tf

dev = std(curve_3_int_big(end, :));

% Mean Absolute Error

MAE_3 = mean(abs(PHI - curve_3_baby));

% Root Mean Square Error

RMSE_3 = rmse(PHI, curve_3_baby);

% Nash-Sutcliffe Efficiency

NSE_3 = 1 - ( sum((PHI - curve_3_baby).^2) / sum((PHI - mean(PHI).^2)));

% Klinge-Gupta Efficiency
% S = sperimentale, O = osservato
% parameters: 
% r = sigma_SO / (sigma_S * sigma_O)
% alpha = sigma_S / sigma_O
% beta = mu_S / mu_O

matrix = cov(curve_3_baby, PHI);
r = matrix(1,2) / (std(curve_3_baby) * std(PHI));
Alpha = std(curve_3_baby) / std(PHI);
Beta = mean(curve_3_baby) / mean(PHI);
KGE_3 = 1 - sqrt((r - 1)^2 + (Alpha - 1)^2 + (Beta - 1)^2);

% R^2 best

RSS = sum((PHI - curve_3_baby).^2);
TSS = sum((PHI - mean(PHI)).^2);
RsquarePHI_3 = 1 - (RSS / TSS);



%% Table with the best results

TAB3 = table(AMIN_3, min(ATHRESHOLD_3_int), ATHRESHOLD_3, max(ATHRESHOLD_3_int), ...
    A_3, B_3, C_3, D_3, K_3, GAMMA_3, ...
    min(curve_3_int_big(end, :)), curve_3(end), max(curve_3_int_big(end, :)), dev, RsquareL_3, RMSE_3, NSE_3, KGE_3, RsquarePHI_3, ...
    'VariableNames',{'Amin','low Athr', 'Athr', 'max Athr', ...
    'a','b','c', 'd','K','gamma', ...
    'min asymp', 'F1', 'max asymp', 'DevSt', 'R2_L','RMSE', 'NSE', 'KGE','R2_PHI' });

% Save the results
% save(strrep("%CATCHMENT%", "%CATCHMENT%", opts.catchment), "Astar","AMIN_3", "ATHRESHOLD_3", ...
%     "A_3", "B_3", "C_3", "D_3", "K_3", "GAMMA_3", "Rsquare", ...
%     "L", "Lw", "Ld", "PHI", "curve_3", "curve_3_baby" ,...
%     "ic90", "area90", "icCAT", "areaCAT");

